#include <stdio.h>
#include <libspe.h>
#include "dma.h"

extern spe_program_handle_t dma_spu0;  // Multiplier
extern spe_program_handle_t dma_spu1;  // Adder

#define NUM_ELEMENTS 1000

// Align data array on cache line (128 bytes) for best DMA performance
int data[NUM_ELEMENTS] __attribute__((aligned(128)));

CONTROL_BLOCK cb __attribute__((aligned(16)));

extern void init_data();
extern void verify_data();

int main()
{
  speid_t id[2];

  // Generate some data
  init_data();

  // Create SPU thread for multiplier. Specifying SPE_MAP_PS allows
  // memory-mapped access to the SPE's MMIO registers (e.g., from other SPEs).
  id[0] = spe_create_thread(0,            // thread group
                            &dma_spu0,    // program
                            (void *)0,            // argp
                            NULL,         // envp
                            -1,           // processor affinity mask (ignore)
                            SPE_MAP_PS);  // flags

  // Create SPU thread for adder
  id[1] = spe_create_thread(0, &dma_spu1, (void *)1, NULL, -1, SPE_MAP_PS);

  // Fill in control block
  for (int i = 0; i < 2; i++) {
    cb.spu_ls[i] = (uintptr32_t)spe_get_ls(id[i]);
    cb.spu_control[i] = (uintptr32_t)spe_get_ps_area(id[i], SPE_CONTROL_AREA);
  }

  cb.num_spus = 2;
  cb.data_addr = (uintptr32_t)data;
  cb.num_elements = NUM_ELEMENTS;

  // Send control block address to all SPUs via mailbox
  for (int i = 0; i < 2; i++) {
    spe_write_in_mbox(id[i], (uintptr32_t)&cb);
  }

  // Spin until SPU 1 tells us it's done by writing to the mailbox
  while (spe_stat_out_mbox(id[1]) == 0);
  // Drain the mailbox
  spe_read_out_mbox(id[1]);

  for (int i = 0; i < 2; i++) {
    spe_wait(id[i], NULL, 0);
  }

  // Check that the SPUs did everything correctly
  verify_data();
  return 0;
}

void init_data()
{
  for (int i = 0; i < NUM_ELEMENTS; i++) {
    data[i] = i;
  }
  printf("Done init\n");
}

void verify_data()
{
  for (int i = 0; i < NUM_ELEMENTS; i++) {
    if (data[i] != i * MUL_FACTOR + ADD_FACTOR) {
      printf("Verify failed!\n");
      return;
    }
  }
  printf("Verify succeeded\n");
}
