#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <libspe.h>
#include "common.h"

// Input points. sizeof(a) == sizeof(b) == 512 bytes.
POINT a[NUM_POINTS] CACHE_ALIGNED;
POINT b[NUM_POINTS] CACHE_ALIGNED;

// Output distances. sizeof(dist) is 16 KB.
float dist[NUM_POINTS][NUM_POINTS] CACHE_ALIGNED;

extern spe_program_handle_t dist_spu;

CONTROL_BLOCK cb[2] QWORD_ALIGNED;

// This function fills the dist array
// dist[i][j] is set to the distance between a[i] and b[j]
void
calc_dist()
{
  speid_t id[2];

  // Set up different control blocks for each SPU
  cb[0].a_addr = (uintptr32_t)a;
  cb[0].b_addr = (uintptr32_t)b;
  cb[0].res_addr = (uintptr32_t)dist;

  cb[1].a_addr = (uintptr32_t)&a[NUM_POINTS / 2];
  cb[1].b_addr = (uintptr32_t)b;
  cb[1].res_addr = (uintptr32_t)&dist[NUM_POINTS / 2];

  // Create SPU threads
  for (int i = 0; i < 2; i++) {
    id[i] = spe_create_thread(0, &dist_spu, &cb[i], NULL, -1, 0);
  }

  // Wait for threads
  for (int i = 0; i < 2; i++) {
    spe_wait(id[i], NULL, 0);
  }
}

extern void gen_points();
extern void verify_dist();

int
main()
{
  // Generate some random points
  gen_points();

  printf("Calculating\n");
  calc_dist();
  printf("Done\n");

  // Check results
  verify_dist();
  return 0;
}

// Fills a and b with random points (co-ords between 0.0 and 10.0)
void
gen_points()
{
  srand(time(NULL));
  for (int i = 0; i < NUM_POINTS; i++) {
    a[i].x = (float)(rand() % 100) / 10.0;
    a[i].y = (float)(rand() % 100) / 10.0;
    b[i].x = (float)(rand() % 100) / 10.0;
    b[i].y = (float)(rand() % 100) / 10.0;
  }
  printf("Generated random points\n");
}

void
verify_dist()
{
  for (int i = 0; i < NUM_POINTS; i++) {
    for (int j = 0; j < NUM_POINTS; j++) {
      if (fabsf(dist[i][j] / distf(&a[i], &b[j]) - 1.0f) > 0.001f) {
        printf("(%.1f, %.1f)-(%.1f, %.1f): expected %.2f, got %.2f\n",
               a[i].x, a[i].y, b[j].x, b[j].y,
               distf(&a[i], &b[j]), dist[i][j]);
        return;
      }
    }
  }
  printf("Correct results\n");
}
