#include <stdio.h>
#include <libspe.h>
#include "common.h"

float mass[NUM_BODIES] CACHE_ALIGNED;  // Mass
VEC3D pos[NUM_BODIES] CACHE_ALIGNED;   // Position vectors
VEC3D vel[NUM_BODIES] CACHE_ALIGNED;   // Velocity vectors

extern spe_program_handle_t sim_spu;

CONTROL_BLOCK cb;
speid_t id[6];

extern uint32_t ticks();
extern float frand();
extern void init_misc();

// Performs one step of the simulation using the SPEs
void
step()
{
  // Tell all SPEs to start computing
  for (int i = 0; i < 6; i++) {
    spe_write_in_mbox(id[i], 0);
  }

  // Wait for all SPEs to indicate they have read body positions
  for (int i = 0; i < 6; i++) {
    while (spe_stat_out_mbox(id[i]) == 0);
    spe_read_out_mbox(id[i]);
  }

  // Tell all SPEs it is safe to write back new positions
  for (int i = 0; i < 6; i++) {
    spe_write_in_mbox(id[i], 0);
  }

  // Wait for all SPEs to indicate they have written back new positions
  for (int i = 0; i < 6; i++) {
    while (spe_stat_out_mbox(id[i]) == 0);
    spe_read_out_mbox(id[i]);
  }
}

// Generates random mass, initial position, initial velocity parameters for all
// bodies
void
init()
{
  for (int i = 0; i < NUM_BODIES; i++) {
    mass[i] = frand() * 5000.f;
    pos[i].x = (frand() - 0.5f) * 1000;
    pos[i].y = (frand() - 0.5f) * 1000;
    pos[i].z = (frand() - 0.5f) * 1000;
    vel[i].x = (frand() - 0.5f) * 10;
    vel[i].y = (frand() - 0.5f) * 10;
    vel[i].z = (frand() - 0.5f) * 10;
  }
}

int
main()
{
  // Initialize timing and RNG
  init_misc();
  // Generate random bodies
  init();

  // Fill in control block
  cb.mass_addr = (uintptr32_t)mass;
  cb.pos_addr = (uintptr32_t)pos;
  cb.vel_addr = (uintptr32_t)vel;

  for (int i = 0; i < 6; i++) {
    id[i] = spe_create_thread(0,
                              &sim_spu,
                              (void *)&cb,  // argp
                              (void *)i,    // envp
                              -1,
                              0);
  }

  while (TRUE) {
    uint32_t t = ticks();

    step();

    printf("%d ms\n", ticks() - t);
  }

  return 0;
}
