#ifndef om_opencl_utils_
#define om_opencl_utils_

#include <stdio.h>
#include <stdlib.h>

#ifndef __USE_POSIX199309
#define __USE_POSIX199309
#include <time.h>
#undef __USE_POSIX199309
#else
#include <time.h>
#endif

typedef struct timespec Timer;

static inline void initTimer(Timer* timer) {
  clock_gettime(CLOCK_MONOTONIC,timer);
}

static inline double getTimer(Timer* timer) {
  struct timespec endTimespec;
  clock_gettime(CLOCK_MONOTONIC,&endTimespec);
  return (endTimespec.tv_sec-timer->tv_sec)+
    (endTimespec.tv_nsec-timer->tv_nsec)*1e-9;
}

static inline double getAndResetTimer(Timer* timer) {
  struct timespec endTimespec;
  clock_gettime(CLOCK_MONOTONIC,&endTimespec);
  double result=(endTimespec.tv_sec-timer->tv_sec)+
    (endTimespec.tv_nsec-timer->tv_nsec)*1e-9;
  *timer=endTimespec;
  return result;
}

static inline double getTimerDifference(Timer* timerStart,Timer* timerEnd) {
  return (timerEnd->tv_sec-timerStart->tv_sec)+
    (timerEnd->tv_nsec-timerStart->tv_nsec)*1e-9;
}

#ifndef NDEBUG

typedef struct {
  cl_int numeric;
  const char* string;
} __OpenclError;

static const __OpenclError __openclErrors[] = {
  { (cl_int)(-1000),                             "unknown error" },
  { CL_SUCCESS,                                  "success" },
  { CL_DEVICE_NOT_FOUND,                         "device not found" },
  { CL_DEVICE_NOT_AVAILABLE,                     "device not available" },
  { CL_COMPILER_NOT_AVAILABLE,                   "compiler not available" },
  { CL_MEM_OBJECT_ALLOCATION_FAILURE,            "memory object allocation failure" },
  { CL_OUT_OF_RESOURCES,                         "out of resources" },
  { CL_OUT_OF_HOST_MEMORY,                       "out of host memory" },
  { CL_PROFILING_INFO_NOT_AVAILABLE,             "profiling info not available" },
  { CL_MEM_COPY_OVERLAP,                         "memory copy overlap" },
  { CL_IMAGE_FORMAT_MISMATCH,                    "image format mismatch" },
  { CL_IMAGE_FORMAT_NOT_SUPPORTED,               "image format not supported" },
  { CL_BUILD_PROGRAM_FAILURE,                    "build program failure" },
  { CL_MAP_FAILURE,                              "map failure" },
  { CL_MISALIGNED_SUB_BUFFER_OFFSET,             "misaligned subbuffer offset" },
  { CL_EXEC_STATUS_ERROR_FOR_EVENTS_IN_WAIT_LIST,"execution status error for events in waiting list" },
  { CL_INVALID_VALUE,                            "invalid value" },
  { CL_INVALID_DEVICE_TYPE,                      "invalid device type" },
  { CL_INVALID_PLATFORM,                         "invalid platform" },
  { CL_INVALID_DEVICE,                           "invalid device" },
  { CL_INVALID_CONTEXT,                          "invalid context" },
  { CL_INVALID_QUEUE_PROPERTIES,                 "invalid queue properties" },
  { CL_INVALID_COMMAND_QUEUE,                    "invalid command queue" },
  { CL_INVALID_HOST_PTR,                         "invalid host pointer" },
  { CL_INVALID_MEM_OBJECT,                       "invalid memory object" },
  { CL_INVALID_IMAGE_FORMAT_DESCRIPTOR,          "invalid image format descriptor" },
  { CL_INVALID_IMAGE_SIZE,                       "invalid image size" },
  { CL_INVALID_SAMPLER,                          "invalid sampler" },
  { CL_INVALID_BINARY,                           "invalid binary" },
  { CL_INVALID_BUILD_OPTIONS,                    "invalid build options" },
  { CL_INVALID_PROGRAM,                          "invalid program" },
  { CL_INVALID_PROGRAM_EXECUTABLE,               "invalid program executable" },
  { CL_INVALID_KERNEL_NAME,                      "invalid kernel name" },
  { CL_INVALID_KERNEL_DEFINITION,                "invalid kernel definition" },
  { CL_INVALID_KERNEL,                           "invalid kernel" },
  { CL_INVALID_ARG_INDEX,                        "invalid argument index" },
  { CL_INVALID_ARG_VALUE,                        "invalid argument value" },
  { CL_INVALID_ARG_SIZE,                         "invalid argument size" },
  { CL_INVALID_KERNEL_ARGS,                      "invalid kernel arguments" },
  { CL_INVALID_WORK_DIMENSION,                   "invalid work dimension" },
  { CL_INVALID_WORK_GROUP_SIZE,                  "invalid work group size" },
  { CL_INVALID_WORK_ITEM_SIZE,                   "invalid work item size" },
  { CL_INVALID_GLOBAL_OFFSET,                    "invalid global offset" },
  { CL_INVALID_EVENT_WAIT_LIST,                  "invalid event wait list" },
  { CL_INVALID_EVENT,                            "invalid event" },
  { CL_INVALID_OPERATION,                        "invalid operation" },
  { CL_INVALID_GL_OBJECT,                        "invalid GL object" },
  { CL_INVALID_BUFFER_SIZE,                      "invalid buffer size" },
  { CL_INVALID_MIP_LEVEL,                        "invalid MIP level" },
  { CL_INVALID_GLOBAL_WORK_SIZE,                 "invalid global work size" },
  { CL_INVALID_PROPERTY,                         "invalid property" }
};

static int __openclErrorCount = 
  sizeof(__openclErrors)/sizeof(__OpenclError);

#define openclVerify(x) do {                                                \
    cl_int __cl_result = x;                                                 \
    if (__cl_result!=CL_SUCCESS) {                                          \
      int __cl_error_index=0;                                               \
      int __cl_i;                                                           \
      for (__cl_i=0;__cl_i<__openclErrorCount;__cl_i++) {                   \
        if (__openclErrors[__cl_i].numeric==__cl_result) {                  \
          __cl_error_index=__cl_i;                                          \
          break;                                                            \
        }                                                                   \
      }                                                                     \
      fprintf(stderr,"%s:%i: error: OpenCL function call failed:\n"         \
                     "  %s;\nmessage: %s (%i)\n",                           \
              __FILE__,__LINE__,#x,__openclErrors[__cl_error_index].string, \
              __cl_result);                                                 \
      exit(1);                                                              \
    }                                                                       \
  } while(0)

#define openclVerifyVal(x) do {                                             \
    cl_int __cl_result = x;                                                 \
    if (__cl_result!=CL_SUCCESS) {                                          \
      int __cl_error_index=0;                                               \
      int __cl_i;                                                           \
      for (__cl_i=0;__cl_i<__openclErrorCount;__cl_i++) {                   \
        if (__openclErrors[__cl_i].numeric==__cl_result) {                  \
          __cl_error_index=__cl_i;                                          \
          break;                                                            \
        }                                                                   \
      }                                                                     \
      fprintf(stderr,"%s:%i: error: OpenCL framework reports problem.\n"    \
                     "message: %s (%i)\n",                                  \
              __FILE__,__LINE__,__openclErrors[__cl_error_index].string,    \
              __cl_result);                                                 \
      exit(1);                                                              \
    }                                                                       \
  } while(0)

#else

#define openclVerify(x) do {                                                \
    x;                                                                      \
  } while(0)

#define openclVerifyVal(x) do {                                             \
    x;                                                                      \
  } while(0)

#endif

static cl_program 
openclCreateProgramFromSourceFile(cl_context context,cl_device_id device,
                                  const char* filename) {

  FILE* file;
  char* sourceString;
  
  file = fopen(filename, "r");
  if (file==NULL) {
    fprintf(stderr, "Error: cannot open kernel file '%s'.\n",filename);
    exit(1);
  }
  
  fseek(file,0,SEEK_END);
  size_t size = ftell(file);
  fseek(file,0,SEEK_SET);
  
  sourceString = (char*)malloc(size);
  fread(sourceString, 1, size, file);
  fclose(file);
  
  cl_int result;
  
  cl_program program 
    = clCreateProgramWithSource(context,1,(const char**)&sourceString, 
                                (const size_t*)&size,&result);
  if (result!=CL_SUCCESS) {
    fprintf(stderr,"Error: cannot create program from source file '%s'.\n",
            filename);
    openclVerify(result);
  }
  
  free(sourceString);
  
  result=clBuildProgram(program,1,&device,NULL,NULL,NULL);

  if (result!=CL_SUCCESS) {
    fprintf(stderr,"Error: cannot build program from source file '%s'.\n",
            filename);
    size_t dummy;
    char logString[65536];
    openclVerify(clGetProgramBuildInfo(program,device,CL_PROGRAM_BUILD_LOG,
                                       65536,logString,&dummy));
    fprintf(stderr,"%s\n",logString);
    openclVerify(result);
  }

  return program;
}

#if __cplusplus >=201100L || defined __GXX_EXPERIMENTAL_CXX0X__

template<typename F> static void 
__openclSetArguments(cl_kernel kernel,int offset,
                     F firstArgument) {

  openclVerify(clSetKernelArg(kernel,offset,sizeof(F),
                              (void*)&firstArgument));
}

template<typename F,typename... T> static void 
__openclSetArguments(cl_kernel kernel,int offset,
                     F firstArgument,T... arguments) {

  openclVerify(clSetKernelArg(kernel,offset,sizeof(F),
                              (void*)&firstArgument));
  __openclSetArguments<T...>(kernel,offset+1,arguments...);
}

template<typename... T> static void 
openclCallKernel(cl_command_queue command_queue,
                 cl_kernel kernel,cl_int dimension,
                 size_t global_item_size[],
                 size_t local_item_size[],T... arguments) {
  __openclSetArguments<T...>(kernel,0,arguments...);
  openclVerify(clEnqueueNDRangeKernel(command_queue,kernel,dimension,NULL,
                                      global_item_size,local_item_size, 
                                      0,NULL,NULL));
}

#endif

#endif
