/*************************************************************************************************
  Example 5            : mpi_sum_reduce_scatter.c 
  Objective            : To find sum of 'n' integers on 'p' processors
                         using MPI collective communication and communciation
                         library call ( MPI_REDUCE ).
                         This example demonstrates the use of 
                         MPI_Init
                         MPI_Comm_rank
                         MPI_Comm_size
                         MPI_Reduce
                         MPI_Finalize
  Input               : Automatic generation of input
                        The rank of each proceess is input on each process.
  Output              : Process with Rank 0 should print the sum of 'n' values.
**************************************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include "mpi.h"

//Constants
#define ARRAY_SIZE 1000
//Macros
#define BLOCK_LOW(id,p,n) ((id)*(n)/(p))
#define BLOCK_HIGH(id,p,n) (BLOCK_LOW((id)+1,p,n)-1)
#define BLOCK_SIZE(id,p,n) (BLOCK_LOW((id)+1,p,n)-BLOCK_LOW(id,p,n))
#define BLOCK_OWNER(index,p,n) (((p)*((index)+1)-1)/(n))

int main(int argc,char *argv[])
{
	int myRank, numProcs;
	int root = 0;
	int globalSum = 0, localSum = 0, serialSum = 0;
	int *elements;
	int *elementsPart;
	int i;
	MPI_Status status;

	//MPI initialization
    MPI_Init(&argc,&argv);
    MPI_Comm_rank(MPI_COMM_WORLD, &myRank);
    MPI_Comm_size(MPI_COMM_WORLD, &numProcs);

	if (myRank == root)
	{
		//Dynamic Memory Allocation
		elements = (int*) malloc(sizeof(int) * ARRAY_SIZE);

		//Array Elements Initialization
		for (i = 0; i < ARRAY_SIZE; i++) 
		{
			elements[i] = i * 2;
		}
		
		//Serial calculation
		for (i = 0; i < ARRAY_SIZE; i++) 
		{
			serialSum += elements[i];
		}
		printf("Serial SUM = %d\n", serialSum);
	}
	
	//Allocate memory for the temporal array
	int arrayPartSize = BLOCK_SIZE(myRank, numProcs, ARRAY_SIZE);
	elementsPart = (int*) malloc(sizeof(int) * arrayPartSize);

	MPI_Scatter(elements, arrayPartSize, MPI_INT, elementsPart, arrayPartSize, MPI_INT, root, MPI_COMM_WORLD);

	//Perform the local calculation
	for (i = 0; i < arrayPartSize; i++)
	{
		localSum += elementsPart[i];
	}

	//The Reduce
    MPI_Reduce(&localSum, &globalSum, 1, MPI_INT, MPI_SUM, root, MPI_COMM_WORLD);

    if (myRank == root)
	{
		printf("SUM = %d\n", globalSum);
	}
	
	//MPI finalization
    MPI_Finalize();
	return 0;
}




