#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include <assert.h>
#include <mpi.h>
#include <math.h>
#include <memory.h>

#include "list.h"
#include "tsp.h"

int myrank, NumProcs, NumCities;

int *Dist;
int numworkers;

Path::Path () { 
    length=0; 
    visited=1;
    for (int i = 0; i < NumCities; i++) {
        city[i]=i;
    }
}

void Path::Set (int len, int *cit, int vis) {
    length = len;
    memcpy (city, cit, NumCities*sizeof(int));
    visited = vis;
}

void Path::Print() {
    for (int i=0; i<visited; i++) {
        printf("  %d", city[i]);
    }
    printf("; length = %d\n", length);
}

void Fill_Dist( void ) {
    FILE *fin; 

	if (myrank == 0) {
        fin = fopen("graph.in", "r");
		if (fin == NULL) {
            printf("Error: File graph.in could not be opened.\n");
            exit(1);
        }
        fscanf(fin, "%d\n", &NumCities);
		fclose(fin);
	}

	assert(NumCities <= MAXCITIES);
    MPI_Bcast(&NumCities, 1, MPI_INT, 0, MPI_COMM_WORLD);
	
	Dist = (int*) malloc(NumCities*NumCities*sizeof(int));
	if (Dist == NULL ) {
		printf ("Error: Not able to allocate memory on process %d.\n", myrank);
		exit(1);
	}

	if (myrank == 0) {
        fin = fopen("graph.in", "r");
        for (int i = 0; i < NumCities; ++i) {
            for (int j = 0; j< NumCities; ++j) {
                fscanf(fin, "%d ", &Dist[i*NumCities + j]);
            }
            fscanf(fin, "\n");
        }
        fclose(fin);

		printf("Number of cities %d in Dist array\n", NumCities);
        for(int i = 0; i < NumCities; i++) {
            for(int j = 0; j < NumCities; j++) {
				printf("%5d", Dist[i*NumCities+j]);
            }
			printf("\n");
		}
		fflush(stdout);
	}

	MPI_Bcast(Dist, NumCities*NumCities, MPI_INT, 0, MPI_COMM_WORLD);
}

void Coordinator() {
    MPI_Status status;
    Msg_t msg;

    int* waiting = new int[NumProcs];     
    int nwait = 0;
    int bpath = 0;                        
                                        
    Path Shortest;
    List queue;
    Path *P = new Path;
    queue.Insert(P, 0);
    Shortest.length = INT_MAX;   

    printf("Coordinator with ID %d started\n", myrank);
	printf("Number of worker tasks will be %d\n", numworkers);
	fflush(stdout);
    
    while (nwait < NumProcs-1) {
        MPI_Recv(&msg, MSGSIZE, MPI_INT, MPI_ANY_SOURCE, MPI_ANY_TAG, MPI_COMM_WORLD, &status); 

        switch (status.MPI_TAG) {
            case BEST_PATH_TAG: 
	            if (msg.length < Shortest.length) {
                    bpath++;
                    printf("Got best path %d, source = %d, length = %d\n", bpath, status.MPI_SOURCE, msg.length);
                    
                    Shortest.Set(msg.length, msg.city, NumCities);
                    for( int i = 1 ; i<NumProcs ; i++ ) {
                        MPI_Send( &(Shortest.length), 1, MPI_INT, i, UPDATE_BEST_PATH_TAG, MPI_COMM_WORLD );
					}
                }
                break;
            case PUT_PATH_TAG:
	            if (nwait > 0) {
                    MPI_Send (&msg, MSGSIZE, MPI_INT, waiting[--nwait], REPLY_PATH_TAG, MPI_COMM_WORLD);
                } else {
                    P = new Path();
                    P->Set(msg.length, msg.city, msg.visited);
                    queue.Insert(P, msg.length);
                }
                break;
            case GET_PATH_TAG:
                if (!queue.IsEmpty()) {
                    P = (Path *)queue.Remove(NULL); 
                    msg.length = P->length;
                    memcpy(msg.city, P->city, MAXCITIES*sizeof(int));
                    msg.visited = P->visited;
                    MPI_Send (&msg, MSGSIZE, MPI_INT, status.MPI_SOURCE, REPLY_PATH_TAG, MPI_COMM_WORLD);
                    delete P;
                } else {
                    waiting[nwait++] = status.MPI_SOURCE;
                    if (nwait == NumProcs-1) {
                        for (int i=1; i<NumProcs; i++) {
                            MPI_Send (NULL, 0, MPI_INT, i, DONE_TAG, MPI_COMM_WORLD);
						}
					}
	            }
	            break;
        }
    }
    printf("Shortest path:\n");
    Shortest.Print();
	fflush(stdout);
}

void Worker() { 
    MPI_Status status;
    Msg_t msg;
    int shortestLength = INT_MAX;

    printf("Worker %d started\n", myrank);
	fflush(stdout);
    MPI_Send(NULL, 0, MPI_INT, 0, GET_PATH_TAG, MPI_COMM_WORLD);

    while (1) {
        MPI_Recv (&msg, MSGSIZE, MPI_INT, 0, MPI_ANY_TAG, MPI_COMM_WORLD, &status);

		if (status.MPI_TAG == DONE_TAG) {
            printf("Worker %d received DONE_TAG\n", myrank);
            break;
        }
    
		if (status.MPI_TAG==UPDATE_BEST_PATH_TAG) {
            shortestLength = msg.length;    
            continue;
        }

        msg.visited++;
        if (msg.visited == NumCities) {
            int d1 = Dist[(msg.city[NumCities-2])*NumCities + msg.city[NumCities-1]];
            int d2 = Dist[(msg.city[NumCities-1])*NumCities ];
            if (d1 * d2) {
                msg.length += d1 + d2;
                if (msg.length < shortestLength) {
	                MPI_Send (&msg, MSGSIZE, MPI_INT, 0, BEST_PATH_TAG, MPI_COMM_WORLD);
				}
            }
        } else {
            int length = msg.length;
            for (int i=msg.visited-1; i<NumCities; i++) {
                //Swap city[i] and city[visted-1]
	            if (i > msg.visited-1) {
                    int tmp = msg.city[msg.visited-1];
                    msg.city[msg.visited-1] = msg.city[i];
                    msg.city[i] = tmp;
                }
      
                //Visit city[visited-1]
                msg.length = length + Dist[(msg.city[msg.visited-2])*NumCities + msg.city[msg.visited-1]];
                if (msg.length < shortestLength) {
                    MPI_Send (&msg, MSGSIZE, MPI_INT, 0, PUT_PATH_TAG, MPI_COMM_WORLD);
                }
            }
        }
        MPI_Send (NULL, 0, MPI_INT, 0, GET_PATH_TAG, MPI_COMM_WORLD);
    }
}

int main(int argc, char *argv[]) {
    int rc = MPI_Init(&argc, &argv);
    rc += MPI_Comm_rank(MPI_COMM_WORLD, &myrank);
    rc += MPI_Comm_size(MPI_COMM_WORLD, &NumProcs);

	double exec_time = 0.0;

    if (rc != MPI_SUCCESS) {
        printf("Error initializing MPI\n");
		return 1;
	} else {
        printf("MPI task ID is %d from %d tasks\n", myrank, NumProcs);
	}

    numworkers = NumProcs-1;
    if (NumProcs < 2) {
        printf("At least 2 processes are required\n");
        return 1;
    }  
    fflush(stdout);
	
	if (myrank==0) {
	    exec_time -= MPI_Wtime();
	}
	//Fill the matrix
	Fill_Dist();  

	//Start the processes
    if (myrank==0) {
        Coordinator();
	} else {
        Worker();
	}
  
	if (myrank==0) {
	    exec_time += MPI_Wtime();
	    printf("Execution time %f sec\n", exec_time);
	}
    MPI_Finalize();
    printf("Rank %d exiting\n", myrank);
    return 0;
}
