package users;

import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;

import javax.servlet.ServletContext;

/**
 * UserManagement manages all kind of users. It keeps the information for
 * registered users/admins. Only one instance of this class should be created
 * and used
 * 
 * @author Radoslav
 * 
 */
public class UserManagement {
	/**
	 * path to admins.txt file which contains the admin users. Path starts from
	 * the context root directory.
	 */
	private static final String admins_file = "/WEB-INF/users/admins.txt";

	/**
	 * path to users.txt file which contains the normal users. Path starts from
	 * the context root directory.
	 */
	private static final String users_file = "/WEB-INF/users/users.txt";

	private Properties users = new Properties();
	private Properties admins = new Properties();

	private ServletContext context;

	public UserManagement(ServletContext context) {
		this.context = context;
	}

	/**
	 * loads all normal users into a property object.
	 */
	public void loadUsers() {
		InputStream is = context.getResourceAsStream(users_file);
		try {
			users.load(is);
		} catch (IOException e) {
			e.printStackTrace();
		} finally {
			try {
				is.close();
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
	}

	/**
	 * loads all admin users into a property object
	 */
	public void loadAdmins() {
		InputStream is = context.getResourceAsStream(admins_file);
		try {
			admins.load(is);
		} catch (IOException e) {
			e.printStackTrace();
		} finally {
			try {
				is.close();
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
	}

	/**
	 * Checks whether the username and password are correct. This method used
	 * when a login is called.
	 * 
	 * @param username
	 *            the username which is passed into the login form
	 * @param password
	 *            the password which is passed into the login form
	 * @return true if username and password are correct. And false otherwise.
	 */
	public boolean checkUser(String username, String password) {
		if (password.equals(users.getProperty(username))) {
			return true;
		}
		if (password.equals(admins.getProperty(username))) {
			return true;
		}
		return false;
	}

	/**
	 * Adds an user into normal users group. This method is invoked whenever
	 * some new user is registered via the registration form
	 * 
	 * @param username
	 *            the username of the newly administrator
	 * @param password
	 *            the passowrd of the newly administrator
	 * @return true if registration is successful and false otherwise.
	 */
	public boolean registerUser(String username, String password) {
		if (users.getProperty(username) == null) {
			users.put(username, password);
			try {
				String pathname = context.getRealPath(users_file);
				System.out.println(pathname);
				users.store(new FileWriter(pathname), "new user added");
			} catch (IOException e) {
				e.printStackTrace();
				users.remove(username);
				return false;
			}
		}
		return false;
	}

	/**
	 * Adds an user into admin group. Only one administrator can add some other
	 * user as administrator
	 * 
	 * @param username
	 *            the username of the newly administrator
	 * @param password
	 *            the passowrd of the newly administrator
	 * @return true if registration is successful and false otherwise.
	 */
	public boolean registerAdmin(String username, String password) {
		if (admins.getProperty(username) == null) {
			admins.put(username, password);
			try {
				String pathname = context.getRealPath(admins_file);
				System.out.println(pathname);
				admins.store(new FileWriter(pathname), "new user added");
			} catch (IOException e) {
				e.printStackTrace();
				admins.remove(username);
				return false;
			}
		}
		return false;
	}

	/**
	 * Checks whether the user is part of the admin group.
	 * 
	 * @param username
	 *            the username that will be checked.
	 * @return true if the user is admin and false otherwise.
	 */
	public boolean isAdminUser(String username) {
		return admins.getProperty(username) != null;
	}

}
